// DAQSys.cpp: implementation of the DAQSys class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Visual C Sample.h"
#include "DAQSys.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
static const int PCI_MIO_16E_1 = 205;
static const int PXI_6070E = 207;
static const int DAQPad_6070E = 275;
static const int AT_MIO_16E_1 = 44;
static const int PCI_6071E = 223;
static const int PXI_6071E = 258;
static const int AT_MIO_16E_2 = 25;
static const int AT_MIO_64E_3 = 38;
static const int DAQCard_6062E = 88;
static const int PCI_6052E = 273;
static const int PXI_6052E = 274;
static const int DAQPad_6052E = 276;
static const int PCI_MIO_16E_4 = 206;
static const int PXI_6040E = 208;
static const int DAQCard_AI_16E_4 = 53;
static const int PCI_MIO_16XE_10 = 204;
static const int PXI_6030E = 209;
static const int AT_MIO_16XE_10 = 50;
static const int PCI_6031E = 220;
static const int PXI_6031E = 259;
static const int PCI_6032E = 221;
static const int AT_AI_16XE_10 = 51;
static const int PCI_6033E = 222;
static const int PCI_6034E = 314;
static const int PXI_6034E = 315;
static const int PCI_6035E = 316;
static const int PXI_6035E = 317;
static const int AT_MIO_16E_10 = 36;
static const int DAQPad_6020E = 76;
static const int AT_MIO_16DE_10 = 37;
static const int PCI_6023E = 267;
static const int DAQCard_6023E = 90;
static const int PXI_6023E = 268;
static const int PCI_6024E = 269;
static const int DAQCard_6024E = 91;
static const int PXI_6024E = 270;
static const int PCI_6025E = 271;
static const int PXI_6025E = 272;
static const int PCI_MIO_16XE_50 = 202;
static const int AT_MIO_16XE_50 = 39;
static const int DAQPad_MIO_16XE_50 = 43;
static const int PXI_6011E = 210;
static const int DAQCard_AI_16XE_50 = 52;

DAQSys::DAQSys()
{
	AIGain = 1;
	scanCount = 2;
	numChannels = 7;
	deviceNumber = 1;
	firstChannel = 0;
	scanRate = 100;
	range = 20.0;
	bipolarity = true;
	setupMinMax(range, bipolarity);
	this->init(1,0,100);
}

int DAQSys::init(short changeDeviceNumber, short changeFirstChannel, double changeScanRate)
{
	for (short i=0; i<numChannels; i++) {
		channels[i] = (short)(i+changeFirstChannel);
	}
	int status = DAQ_Clear( deviceNumber ); /*first stop any acquisition that is already 
												happening*/

	status = SCAN_Setup(changeDeviceNumber, numChannels, channels, channel_gains);
	// positive values may be allowed...?
#if TEST_WITHOUT_DAQ
	random_gen = false;
	if (status != 0) {
		random_gen = true;
		status = 0;
		srand(2131978);
	}
#else
	if (status != 0) {
		// revert to original settings
		for (short i=0; i<numChannels; i++) {
			channels[i] = (short)(i+firstChannel);
		}
		status = SCAN_Setup(deviceNumber, numChannels, channels, channel_gains);
		if (status != 0) {
			// if can not revert to original settings, throw exception
			DAQException* error = new DAQException(status);
			throw error;
		}
	}
	/* by asking the DAQ how many volts a single count is, we can get
	   the conversion factor for ourselves */
	deviceNumber = changeDeviceNumber;
	firstChannel = changeFirstChannel;
	scanRate = changeScanRate;
#endif
	setupMinMax(range, bipolarity);	
	unsigned int sampleCount = scanCount*numChannels;
	
	double sampleRate = numChannels*scanRate;
	double scanTime = (sampleCount/sampleRate)/0.055;	// in number of ticks
	int num_ticks = 2*(int)(scanTime)+10;				// the number of timer ticks (55 ms) before timing out
	status = Timeout_Config(deviceNumber, num_ticks);
	short sampleTimeBase;
	unsigned short sampleInterval;
	status = DAQ_Rate( sampleRate, 0, &sampleTimeBase, &sampleInterval );
	//daq_status = SCAN_Op(deviceNumber, numChannels, channels, channel_gains, bin_data, sampleCount, sampleRate, 0);
	status = DAQ_DB_Config( deviceNumber, 1 );
	status = SCAN_Start( deviceNumber, (short *)scanBuffer, SCAN_BUFFER_SIZE, sampleTimeBase, sampleInterval, sampleTimeBase, 0 );
	return status;
}

DAQSys::~DAQSys()
{
	DAQ_Clear( deviceNumber );
}

double DAQSys::ScanGauge(short chan)
{
	short status;
	double voltage;
	status = AI_VRead (deviceNumber, chan, AIGain, &voltage);	// read volts
//	short iIgnoreWarning = 0;
//	NIDAQErrorHandler (status, "AI_VRead", iIgnoreWarning);
	
	return voltage;
}

int DAQSys::ScanGauges(double voltages[7], bool useStoredValues)
{
	static double old_voltages[7] = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };
	static int status = 1;
	if (useStoredValues && status != 1) {
		for (int i=0; i<7; i++) {
			voltages[i] = old_voltages[i];
		}
		return status;
	}

#if TEST_WITHOUT_DAQ
	if (random_gen) {
		static int binary[7];
		if (status == 1) {
			for (int i=0; i<7; i++) {
				int r = rand();
				int x = r%(2<<11);	// [    0, 4095]
				int y = x-(2<<10);	// [-2048, 2047]
				binary[i] = y;
				double v = binary[i]*voltsPerCount;
				voltages[i] = v;
				old_voltages[i] = v;
			}
//			binary[6] = 0;
//			voltages[6] = 0.0;
//			old_voltages[6] = 0.0;
			status = 0;
			return 0;
		}
		status = 0;
		for(int i=0; i<7; i++) {
			int r = rand();
			int x = r%101;
			int y = x-50;
			int b = binary[i];
			b += y;
			binary[i] = b;
			double v = b*voltsPerCount;
			if (binary[i] <= AIMin) {
				v = AIMin*voltsPerCount;
				status = -1;
			}
			if (binary[i] >= AIMax) {
				v = AIMax*voltsPerCount;
				status = -1;
			}
			voltages[i] = v;
			old_voltages[i] = v;
		}
		return status;
	}
#endif
	status = 0;
	short daq_status = 0;
	short* bin_data = new short[scanCount * numChannels];
//	for (int i=0; i<7; i++) {
//		bin_data[i] = 13*i;
//	}
	unsigned long newestPointIndex;
	short daqStopped;
	do{
		daq_status = DAQ_Monitor( deviceNumber, -1, 0, scanCount * numChannels, bin_data, &newestPointIndex, &daqStopped );
	}while ( ( overWriteError == daq_status ) || ( dataNotAvailError == daq_status ) );
	//daq_status = SCAN_Op(deviceNumber, numChannels, channels, channel_gains, bin_data, sampleCount, sampleRate, 0);
//	}
/*
	catch (void* e)
	{
		status = 1;
	}
	catch (...)
	{
		status = 1;
	}
*/
	// positive values may be allowed...?
	if (daq_status != 0) {
		DAQException* error = new DAQException(daq_status);
		throw error;
	}

	for (unsigned int channel=0; channel<numChannels; channel++) {
		int channel_sum = 0;
		for (unsigned int scan=0; scan<scanCount; scan++) {
			if (bin_data[numChannels*scan + channel] <= AIMin) {
				bin_data[numChannels*scan + channel] = (short)(AIMin);
				status = -1;
			}
			if (bin_data[numChannels*scan + channel] >= AIMax) {
				bin_data[numChannels*scan + channel] = (short)(AIMax);
				status = -1;
			}
			channel_sum += bin_data[numChannels*scan + channel];
		}
		double t = 1.0*channel_sum/scanCount*voltsPerCount;
		voltages[channel] = t;
		old_voltages[channel] = t;
	}

	delete[] bin_data;

	return status;
}

#define xx		0		/* placeholder - gain not supported */
#define xxx		0		/* placeholder - gain not supported */
#define xxxx	0		/* placeholder - gain not supported */
#define x0_5	1		/* gain of 0.5 volts is supported */
#define x1		2		/* gain of 1 is supported */
#define x2		4		/* gain of 2 is supported */
#define x5		8		/* gain of 5 is supported */
#define x10		16		/* gain of 10 is supported */
#define x20		32		/* gain of 20 is supported */
#define x50		64		/* gain of 50 is supported */
#define x100	128		/* gain of 100 is supported */

void DAQSys::setupMinMax(double DesiredRange, bool bipolar)
{
//	DesiredRange = DesiredRange/2;
	const double gains[8] = { 0.5, 1, 2, 5, 10, 20, 50, 100 };
	const int  aigains[8] = {  -1, 1, 2, 5, 10, 20, 50, 100 };
	range = DesiredRange;
	bipolarity = bipolar;
	unsigned long info;
	int Res;
	int MaxScanRate;
	int BPBase;
	int UPBase;
#if TEST_WITHOUT_DAQ
	info = DAQPad_6020E;
	voltsPerCount = 10.0/(2<<10);
#else
	Get_DAQ_Device_Info(deviceNumber, ND_DEVICE_TYPE_CODE, &info);
#endif
	unsigned char BPGains;
	unsigned char UPGains;
	switch (info) {
        case PCI_MIO_16E_1:	case PXI_6070E:		case DAQPad_6070E:
		case AT_MIO_16E_1:	case PCI_6071E:		case PXI_6071E:
		case AT_MIO_16E_2:	case AT_MIO_64E_3:	case DAQCard_6062E:
		case PCI_MIO_16E_4:	case PXI_6040E:		case DAQCard_AI_16E_4:
		case AT_MIO_16E_10:	case DAQPad_6020E:	case AT_MIO_16DE_10:
            Res = 12;
            MaxScanRate = 10000;
            BPBase = 10;
            UPBase = 10;
            BPGains = x0_5 | x1 | x2 | x5 | x10 | x20 | x50 | x100;
            UPGains = xxxx | x1 | x2 | x5 | x10 | x20 | x50 | x100;
			break;
        case PCI_6023E:		case PCI_6024E:		case DAQCard_6024E:
		case PCI_6025E:		case PXI_6025E:		case PXI_6023E:
		case DAQCard_6023E:	case PXI_6024E:
            Res = 12;
            MaxScanRate = 10000;
            BPBase = 10;
            UPBase = 0;
            BPGains = x0_5 | x1 | xx | xx | x10 | xxx | xxx | x100;
            UPGains = xxxx | xx | xx | xx | xxx | xxx | xxx | xxxx;
			break;
		case PCI_MIO_16XE_50:		case AT_MIO_16XE_50:
		case DAQCard_AI_16XE_50:	case DAQPad_MIO_16XE_50:
		case PXI_6011E:
            Res = 16;
            MaxScanRate = 2800;
            BPBase = 20;
            UPBase = 10;
            BPGains = xxxx | x1 | x2 | xx | x10 | xxx | xxx | x100;
            UPGains = xxxx | x1 | x2 | xx | x10 | xxx | xxx | x100;
			break;
        case PCI_MIO_16XE_10:		case PXI_6030E:
		case AT_MIO_16XE_10:		case PCI_6031E:
		case PXI_6031E:				case PCI_6032E:
		case AT_AI_16XE_10:			case PCI_6033E:
		case PCI_6052E:				case PXI_6052E:
		case DAQPad_6052E:
            Res = 16;
            MaxScanRate = 10000;
            BPBase = 20;
            UPBase = 10;
            BPGains = xxxx | x1 | x2 | x5 | x10 | x20 | x50 | x100;
            UPGains = xxxx | x1 | x2 | x5 | x10 | x20 | x50 | x100;
			break;
		case PCI_6034E:				case PCI_6035E:
		case PXI_6034E:				case PXI_6035E:
            Res = 16;
            MaxScanRate = 10000;
            BPBase = 10;
            UPBase = 0;
            BPGains = x0_5 | x1 | xx | xx | x10 | xxx | xxx | x100;
            UPGains = xxxx | xx | xx | xx | xxx | xxx | xxx | xxxx;
			break;
		default:
			throw new DAQException("Unknown Device");
	}
	int i;
	double gain = -1;
	int AIGain = 0;
	for (i=7; i>=0; i--) {
		if (bipolar) {
			if (BPGains & 1<<i) {
				// This gain is allowed
				// If BPBase / BPGains(i) >= DesiredRange Then Exit For
				if ((BPBase / gains[i]) >= DesiredRange) {
					gain = gains[i];
					AIGain = aigains[i];
					break;
				}
			}
		}
		else {
			if (UPGains & 1<<i) {
				// This gain is allowed
				// If BPBase / BPGains(i) >= DesiredRange Then Exit For
				if ((UPBase / gains[i]) >= DesiredRange) {
					gain = gains[i];
					AIGain = aigains[i];
					break;
				}
			}
		}
	}
	if (gain < 0 || AIGain == 0) {
		// we've had a problem
		// TODO: error handler
		throw new DAQException("Unable to find a suitable gain");
		return;
	}
	if (bipolar) {
		int resolution = 2<<(Res-2);
		AIMax = (int)( resolution * DesiredRange / (BPBase/gain) );
		AIMin = -AIMax;
		AIMax = (AIMax == resolution)?resolution-1:AIMax;
	}
	else {
		int resolution = 2<<(Res-1);
		AIMin = 0;
		AIMax = (int)( resolution * DesiredRange / (BPBase/gain) );
		AIMax = (AIMax == resolution)?resolution-1:AIMax;
	}
	AI_VScale(deviceNumber, 0, (short)AIGain, 1.0, 0.0, 1, &voltsPerCount);
	for (short j=0; j<numChannels; j++) {
		channel_gains[j] = (short)(AIGain);
	}
}
