VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
  Persistable = 0  'NotPersistable
  DataBindingBehavior = 0  'vbNone
  DataSourceBehavior  = 0  'vbNone
  MTSTransactionMode  = 0  'NotAnMTSObject
END
Attribute VB_Name = "DAQSystem"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = True
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = False
'Class DAQSystem

'This class module abstracts all of the hardware communication routines.
'It is intended for use with NI-DAQ 6.8 or later.
'To use this program with non-NI-DAQ hardware, implement the public
'interface of this module using calls to your hardware device driver.

'This class sets up NI-DAQ hardware for F/T Analog systems, then retrieves
'individual 7-channel records.  Currently, it supports two methods of operation.
'In one method, AI_Read_Scan is called to acquire the data record.  In
'WindowsNT/2000, however, this method is somewhat slow for a realtime display.
'The second method offers better performance, but is slightly more complex.
'This method uses a continuous double-buffering operation to eliminate
'unnecessary hardware I/O, and calls DAQ-Monitor to retrieve the most recent
'scan.

Option Explicit

'ATI supported devices:
Private Const PCI_MIO_16E_1 = 205
Private Const PXI_6070E = 207
Private Const DAQPad_6070E = 275
Private Const AT_MIO_16E_1 = 44
Private Const PCI_6071E = 223
Private Const PXI_6071E = 258
Private Const AT_MIO_16E_2 = 25
Private Const AT_MIO_64E_3 = 38
Private Const DAQCard_6062E = 88
Private Const PCI_6052E = 273
Private Const PXI_6052E = 274
Private Const DAQPad_6052E = 276
Private Const DAQPad_6052E_1 = 332 'DAQPad-6052E for USB Mass Termination/BNC
Private Const DAQPad_6052E_2 = 361 'DAQPad-6052E for 1394 (BNC)
Private Const DAQPad_6052E_3 = 362 'DAQPad-6052E for Ethernet (BNC)
Private Const DAQPad_6052E_4 = 363 'DAQPad-6052E for Ethernet (Mass Termination)
Private Const PCI_MIO_16E_4 = 206
Private Const PXI_6040E = 208
Private Const DAQCard_AI_16E_4 = 53
Private Const PCI_MIO_16XE_10 = 204
Private Const PXI_6030E = 209
Private Const AT_MIO_16XE_10 = 50
Private Const PCI_6031E = 220
Private Const PXI_6031E = 259
Private Const PCI_6032E = 221
Private Const AT_AI_16XE_10 = 51
Private Const PCI_6033E = 222
Private Const PCI_6034E = 314
Private Const PXI_6034E = 315
Private Const PCI_6035E = 316
Private Const PXI_6035E = 317
Private Const PCI_6036E = 348 'DAQCard-6036E/PCI-6036E
Private Const DAQPad_6036E = 359 'DAQPad-6036E for USB (BNC)
Private Const AT_MIO_16E_10 = 36
Private Const DAQPad_6020E = 76
Private Const AT_MIO_16DE_10 = 37
Private Const PCI_6023E = 267
Private Const DAQCard_6023E = 90
Private Const PXI_6023E = 268
Private Const PCI_6024E = 269
Private Const DAQCard_6024E = 91
Private Const PXI_6024E = 270
Private Const DAQPad_6024E_1 = 354 'DAQPad-6024E for USB (BNC)
Private Const DAQPad_6024E_2 = 356 'DAQPad-6024E for USB (Mass Termination)
Private Const PCI_6025E = 271
Private Const PXI_6025E = 272
Private Const DAQPad_6025E_1 = 357 'DAQPad-6025E for USB (BNC)
Private Const DAQPad_6025E_2 = 358 'DAQPad-6025E for USB (Mass Termination)
Private Const PCI_MIO_16XE_50 = 202
Private Const AT_MIO_16XE_50 = 39
Private Const DAQPad_MIO_16XE_50 = 43
Private Const PXI_6011E = 210
Private Const DAQCard_AI_16XE_50 = 52


Private MaxScanRate As Double       'Maximum scan rate for DAQ F/T system with this card
Private AIGain As Double            'Gain setting to achieve +/-5V input range; depends on device type
Private AIMin As Long               'minimum binary reading; used to check for ADC saturation; depends on device type
Private AIMax As Long               'maximum binary reading; used to check for ADC saturation; depends on device type
Private mDevice As Integer          'NI-DAQ device number; 1 by default
Private mConfigured As Boolean      'indicates whether or not the Setup routine has successfully completed
Private mBuffering As Boolean       'indicates whether double-buffering is currently active
Private buffer() As Integer         'used in buffering mode
Private binary() As Integer         'used in buffering mode
Private mFilterLevel As Integer     'number of records to average (buffering mode only)
Private mVoltConv As Double         'scaling factor from binary to volts (volts/count)
Private mMaxVoltage As Double       'highest acceptable differential voltage
Private mOutputPolarity As Integer  '0=bipolar; 1=unipolar
Private mFirstChannel As Integer    'index of first channel; typically 0
Private mInputRange As Double       'the input range set by the user (total)
Private mCh6Saturates As Boolean    'can a saturation condition be raised by Ch6?


'public interface
Public Property Get MaxVoltage() As Double
    MaxVoltage = mMaxVoltage
End Property
Public Property Get Configured() As Boolean
'indicates whether the Setup routine has completed successfully.
    Configured = mConfigured
End Property
Public Property Get Device() As Integer
'returns the active NI-DAQ device number.  Set using the Setup method.
    Device = mDevice
End Property
Public Property Get BufMode() As Boolean
'returns True if the double-buffering operation is in progress.
    BufMode = mBuffering
End Property
Public Property Get InputRange() As String
    InputRange = "+/-" & mInputRange / 2 & " V"
End Property
Public Property Get Ch6Saturates() As Boolean
    Ch6Saturates = mCh6Saturates
End Property
Public Property Let Ch6Saturates(NewValue As Boolean)
    mCh6Saturates = NewValue
End Property

Public Sub Setup(Optional Device As Integer = 1, Optional BufMode As Boolean = False, _
    Optional FilterLevel As Integer = 0, Optional ScanRate As Double = 10000, _
    Optional InputMode As Integer = 0, Optional InputRange As Double = 20, Optional InputBipolar As Boolean = True, _
    Optional FirstChannel As Integer = 0, Optional Ch6Saturates As Boolean = False)
    'default input mode is +/-10 V, differential
    
    mConfigured = False                 'if errors occur, mConfigured should be false
    
    'turn buffer mode off (in case this is not the first time Setup is called)
    If mBuffering Then StopBuffering    'use old value of mDevice
    
    mDevice = Device
    mFirstChannel = FirstChannel
    mMaxVoltage = InputRange
    If InputBipolar Then mMaxVoltage = mMaxVoltage / 2
    
    If FilterLevel = 0 Or BufMode = False Then FilterLevel = 1
    mFilterLevel = FilterLevel
    
    'setup channels on input ranges depending DAQ hardware and input ranges
    SetInputRanges InputMode, InputRange, InputBipolar
    mInputRange = InputRange
    
    SetupScan BufMode   'setup the scan sequence
    mConfigured = True
    If BufMode Then
        If ScanRate > MaxScanRate / 7 Then
            mConfigured = False
            Err.Raise vbObjectError + 600, , "Scan Rate Too High for this DAQ System"
        End If
        StartBuffering ScanRate
    Else
        ReDim binary(0 To 6) As Integer
    End If
    
    mCh6Saturates = Ch6Saturates
    
    Exit Sub
End Sub
Public Function Scan(ByRef voltages() As Double) As Integer
'This function performs single scan operation or a buffer monitor operation
'Return values
'0: Successful scan or monitor
'1: ADC saturation occurred (it is not valid to compute forces and torques)

    Dim status As Integer           'return value from NI-DAQ functions
    
    If mBuffering = False Then
        'general-purpose one-shot scan; runs slowly on WinNT
        status = AI_Read_Scan(mDevice, binary(0))
        NIDAQErrorHandler status
    Else
        'monitor the current buffer operation
        Dim iChan As Integer
        Dim sequential As Integer
        Dim numPts As Long
        Dim newestPtIndex As Long
        Dim DAQStopped As Integer
        iChan% = -1     'scan all configured channels
        sequential = 0  'grab most recent data
        numPts = mFilterLevel * 7    'number of records to grab (7 samples per record)
        Do
            status = DAQ_Monitor(mDevice, iChan%, sequential, numPts, binary(0), newestPtIndex, DAQStopped)
            'if data is not yet available, or buffer overflow occurs, try again.  These are non-critical errors.
        Loop Until status <> overWriteError And status <> dataNotAvailError 'And status <> overFlowError
        If status = overFlowError Then
            Scan = 2
            Exit Function
        End If
        NIDAQErrorHandler status
    End If
        
    'check for gauge saturation
    Scan = 0 'success
    Dim i As Integer, j As Integer
    
    Dim Sum As Long
    For i = 0 To 6
        Sum = 0
        For j = 0 To mFilterLevel - 1
            If (i <> 6) Or (i = 6 And mCh6Saturates = True) Then
                If binary(i + j * 7) <= AIMin Or binary(i + j * 7) >= AIMax Then Scan = 1 'saturation
            End If
            Sum = Sum + binary(i + j * 7)
        Next j
        voltages(i) = Sum / mFilterLevel * mVoltConv
    Next i
End Function


'private functions
Private Sub SetupScan(BufMode As Boolean)
'Sets up the scan sequence and other configuration options
    Dim status As Integer
    
    'set a timeout for synchronous NI-DAQ functions
    Dim lTimeout As Long
    lTimeout = 180          '180 timer ticks * 55 ms/tick = 10 s
    status = Timeout_Config(mDevice, lTimeout&)
    NIDAQErrorHandler status
    
    'set double-buffering mode as appropriate
    SetBufferMode BufMode
    
    'setup the scan sequence
    Dim i As Integer
    Dim NumChans As Integer
    Dim ChanVector(0 To 6) As Integer
    Dim GainVector(0 To 6) As Integer
    NumChans = 7
    For i = 0 To 6
        ChanVector(i) = i + mFirstChannel
        GainVector(i) = AIGain
    Next i
    status = SCAN_Setup(mDevice, NumChans, ChanVector(0), GainVector(0))
    NIDAQErrorHandler status
    
End Sub
Private Sub ConfigureChannel(iChan As Integer, InputMode As Integer, InputPolarity As Integer)
'Configures one channel for F/T Analog systems (differential inputs with +/-5V range)
    Dim status As Integer
    Dim iRetVal As Integer
    Dim iDriveAIS As Integer
    Dim iInputRange As Integer
    Dim iIgnoreWarning As Integer

    iInputRange = 10 'not used
    iDriveAIS = 0    'not used

    status = AI_Configure(mDevice, iChan%, InputMode%, iInputRange%, InputPolarity%, iDriveAIS%)
    NIDAQErrorHandler status
End Sub
Private Sub SetInputRanges(ByVal InputMode As Integer, ByVal DesiredRange As Double, ByVal DesiredBipolar As Boolean)
'Reads the device type, computes gain and saturation values, and scaling factors
'Configures each channel for the appropriate input mode and range
    Dim status As Integer
    Dim infoValue As Long
    Dim BPGains As Variant
    Dim UPGains As Variant
    Dim BPBase As Double
    Dim UPBase As Double
    Dim Res As Integer

    status = Get_DAQ_Device_Info(mDevice, ND_DEVICE_TYPE_CODE, infoValue)
    NIDAQErrorHandler status
    Select Case infoValue
        Case PCI_MIO_16E_1, PXI_6070E, DAQPad_6070E, _
            AT_MIO_16E_1, PCI_6071E, PXI_6071E, _
            AT_MIO_16E_2, AT_MIO_64E_3, DAQCard_6062E, _
            PCI_MIO_16E_4, PXI_6040E, DAQCard_AI_16E_4, _
            AT_MIO_16E_10, DAQPad_6020E, AT_MIO_16DE_10:
            Res = 12
            BPGains = Array(0.5, 1, 2, 5, 10, 20, 50, 100)
            BPBase = 10
            UPGains = Array(1, 2, 5, 10, 20, 50, 100)
            UPBase = 10
        Case PCI_6023E, PCI_6024E, DAQCard_6024E, PCI_6025E, PXI_6025E, _
            PXI_6023E, DAQCard_6023E, PXI_6024E, DAQPad_6024E_1, _
            DAQPad_6024E_2, DAQPad_6025E_1, DAQPad_6025E_2:
            Res = 12
            BPGains = Array(0.5, 1, 10, 100)
            BPBase = 10
            UPGains = Array()
            UPBase = 0
        Case PCI_MIO_16XE_50, AT_MIO_16XE_50, DAQCard_AI_16XE_50, _
            DAQPad_MIO_16XE_50, PXI_6011E:
            Res = 16
            BPGains = Array(1, 2, 10, 100)
            BPBase = 20
            UPGains = Array(1, 2, 10, 100)
            UPBase = 10
        Case PCI_MIO_16XE_10, PXI_6030E, AT_MIO_16XE_10, PCI_6031E, _
            PXI_6031E, PCI_6032E, AT_AI_16XE_10, PCI_6033E, _
            PCI_6052E, PXI_6052E, DAQPad_6052E, DAQPad_6052E_1, _
            DAQPad_6052E_2, DAQPad_6052E_3, DAQPad_6052E_4:
            Res = 16
            BPGains = Array(1, 2, 5, 10, 20, 50, 100)
            BPBase = 20
            UPGains = Array(1, 2, 5, 10, 20, 50, 100)
            UPBase = 10
        Case PCI_6034E, PCI_6035E, PXI_6034E, PXI_6035E, PCI_6036E, DAQPad_6036E:
            Res = 16
            BPGains = Array(0.5, 1, 10, 100)
            BPBase = 10
            UPGains = Array()
            UPBase = 0
        Case Else:
            Err.Raise vbObjectError + 600, , "Unsupported device"
    End Select
    Select Case infoValue
        Case PCI_MIO_16E_1, PXI_6070E, DAQPad_6070E, _
            AT_MIO_16E_1, PCI_6071E, PXI_6071E
            MaxScanRate = 1250000
        Case AT_MIO_16E_2, DAQCard_6062E:
            MaxScanRate = 500000
        Case AT_MIO_64E_3, PCI_6052E, PXI_6052E, DAQPad_6052E_1, _
            DAQPad_6052E_2, DAQPad_6052E_3, DAQPad_6052E_4:
            MaxScanRate = 333000
        Case PCI_MIO_16E_4, PXI_6040E, DAQCard_AI_16E_4:
            MaxScanRate = 250000
        Case PCI_6034E, PCI_6035E, PCI_6023E, PCI_6024E, DAQCard_6024E, _
            PCI_6025E, PXI_6025E, DAQPad_6024E_1, DAQPad_6024E_2, _
            DAQPad_6025E_1, DAQPad_6025E_2, PCI_6036E, DAQPad_6036E:
            MaxScanRate = 200000
        Case PCI_MIO_16XE_10, PXI_6030E, AT_MIO_16XE_10, PCI_6031E, _
            PXI_6031E, PCI_6032E, AT_AI_16XE_10, PCI_6033E, _
            AT_MIO_16E_10, DAQPad_6020E, AT_MIO_16DE_10:
            MaxScanRate = 100000
        Case PCI_MIO_16XE_50, AT_MIO_16XE_50, DAQCard_AI_16XE_50:
            MaxScanRate = 20000
        Case Else:
            Err.Raise vbObjectError + 600, , "Unsupported device"
    End Select
    
    Dim i As Integer

    'determine best range setting
    If (DesiredBipolar) Then
        For i = UBound(BPGains) To 0 Step -1
            If BPBase / BPGains(i) >= DesiredRange Then Exit For
        Next i
        If i = -1 Then Err.Raise vbObjectError + 600, , "Input range not supported by this device."
        mOutputPolarity = 0 'bipolar
        AIMin = -2 ^ (Res - 1) * DesiredRange / (BPBase / BPGains(i))
        AIMax = 2 ^ (Res - 1) * DesiredRange / (BPBase / BPGains(i))
        If AIMax > 2 ^ (Res - 1) - 1 Then AIMax = 2 ^ (Res - 1) - 1
        If BPGains(i) = 0.5 Then
            AIGain = -1
        Else
            AIGain = BPGains(i)
        End If
    Else
        For i = UBound(UPGains) To 0 Step -1
            If UPBase / UPGains(i) >= DesiredRange Then Exit For
        Next i
        If i <> -1 Then
            mOutputPolarity = 1 'unipolar
            AIMin = 0
            AIMax = 2 ^ Res * DesiredRange / (UPBase / UPGains(i))
            If AIMax > 2 ^ Res - 1 Then AIMax = 2 ^ Res - 1
            If UPGains(i) = 0.5 Then
                AIGain = -1
            Else
                AIGain = UPGains(i)
            End If
        Else
            'see if a bipolar range will work
            For i = UBound(BPGains) To 0 Step -1
                If BPBase / (BPGains(i) * 2) >= DesiredRange Then Exit For
            Next i
            If i = -1 Then Err.Raise vbObjectError + 600, , "Input range not supported by this device."
            mOutputPolarity = 0 'bipolar
            AIMax = (2 ^ (Res - 1)) * DesiredRange / (BPBase / (BPGains(i) * 2))
            If AIMax > 2 ^ (Res - 1) - 1 Then AIMax = 2 ^ (Res - 1) - 1
            AIMin = 0
            If BPGains(i) = 0.5 Then
                AIGain = -1
            Else
                AIGain = BPGains(i)
            End If
        End If
    End If
    
    'configure each channel for F/T transducers
    For i = mFirstChannel To mFirstChannel + 6
        ConfigureChannel i, InputMode, mOutputPolarity
    Next i
    
    'there is a bug in NI-DAQ that causes VScale to return incorrect values
    'in Unipolar mode.  The next command causes the correct value to be
    'returned; not sure why.
    Dim r As Integer
    AI_Read mDevice, mFirstChannel, AIGain, r
    
    'get mVoltConv (coef to scale data from binary to actual voltages)
    '(all channels have the same coefficient)
    status = AI_VScale(mDevice, mFirstChannel, AIGain, 1, 0, 1, mVoltConv)
    NIDAQErrorHandler status
    Exit Sub
End Sub
Private Sub SetBufferMode(DBOn As Boolean)
    'enabled continuous (double-buffering) mode
    Dim status As Integer
    Dim DBMode As Integer
    If DBOn Then
        DBMode = 1  'enable double-buffering mode
    Else
        DBMode = 0  'disable double-buffering mode
    End If
    status = DAQ_DB_Config(mDevice, DBMode)
    NIDAQErrorHandler status
End Sub
Private Sub StartBuffering(ScanRate As Double)
'Starts a double-buffering operation
'Note: the values here set a scan rate near 10kHz, the maximum rated rate
'for ATI Analog F/T systems.
'The actual sampling rate is 69444 Hz/7 channels=9920 Hz.
    Dim Count As Long
    Dim status As Integer
    Dim sampTimeBase As Integer
    Dim sampInterval As Integer
    Dim scanTimeBase As Integer
    Dim scanInterval As Integer
    Count = mFilterLevel * 14           'use 2 buffers each large enough for mFilterLevel records
    If Count < 280 Then Count = 280     'minimum buffer size=20 records
    status = DAQ_Rate(ScanRate * 7, 0, sampTimeBase, sampInterval)
    NIDAQErrorHandler status

    scanTimeBase = sampTimeBase
    scanInterval = 0    'number of clock cycles per scan (0=wait for last sample)
    
    'setup data double-buffer
    ReDim buffer(0 To Count - 1) As Integer     'main DAQ buffer
    ReDim binary(0 To Count / 2 - 1) As Integer    'used with DAQ_Monitor
    
    status = SCAN_Start(mDevice, buffer(0), Count, sampTimeBase, sampInterval, scanTimeBase, scanInterval)
    NIDAQErrorHandler status
    mBuffering = True
End Sub
Private Sub StopBuffering()
'Stops a double-buffering operation
    Dim status As Integer
    status = DAQ_Clear(mDevice)
    NIDAQErrorHandler status
    mBuffering = False
End Sub
Private Sub NIDAQErrorHandler(status As Integer)
'Raises an error based on NI-DAQ return values
    If status <> 0 Then
        Select Case status
            Case unknownDeviceError: Err.Raise vbObjectError + 2, , "NI-DAQ cannot communicate with the specified hardware device.  Make sure the device is compatible with this version of NI-DAQ, and that its device number matches the one in Measurement & Automation Explorer."
            Case 1: Err.Raise "NI-DAQ not installed.  Please install before running this application."
            Case Else: Err.Raise vbObjectError + 1, , "NI-DAQ Error " & status
        End Select
    End If
End Sub
Private Sub Class_Initialize()
    mDevice = 1     'default device number
End Sub
Private Sub Class_Terminate()
    'stop the double-buffering operation to avoid memory errors
    If mBuffering Then StopBuffering
End Sub
